define([
    'underscore',
    'backbone',
    'app-resources',
    'modules/appointments/appointments-module',
    'modules/new-appointment-request/views/common/type-of-care/layout-view',
    'modules/new-appointment-request/tests/helpers/facility-helper',
], function(_, Backbone, resources, module, View, helpers) {
    'use strict';


    function resourceSpy() {
        spyOn(resources, 'get').and.callFake(function() {
            return {formatHref: _.noop};
        });
        return resources.get;
    }


    function fetchSpy(data) {
        spyOn(Backbone.Collection.prototype, 'fetch').and.callFake(function() {
            var deferred = new $.Deferred();
            this.set(data, {parse: true});
            deferred.resolve(this);
            return deferred;
        });
        return module.fetchCollection;
    }


    function mockData() {
        return [
            {
                'id': '203',
                'name': 'Audiology',
                'objectType': 'TypesOfCare',
                'link': [],
            },
            {
                'id': '323',
                'name': 'Primary Care',
                'objectType': 'TypesOfCare',
                'link': [],
            },
            {
                'id': '408',
                'name': 'Optometry',
                'objectType': 'TypesOfCare',
                'link': [],
            },
            {
                'id': '502',
                'name': 'Outpatient Mental Health',
                'objectType': 'TypesOfCare',
                'link': [],
            },
            {
                'id': 'CR1',
                'name': 'Express Care',
                'objectType': 'TypesOfCare',
                'link': [],
            },
        ];
    }


    describe('Step 1 Type of Care', function() {


        /**
         * https://issues.mobilehealth.DOMAIN.EX/browse/VAR-9195
         *
         * As a user
         * I want to be able to select the specific location where I want to receive care
         * So that my booking process is streamlined focuses on a smaller number of specific clinics
         *
         * When a user views the New Appointment/Request form, the “routing questions” shown at the top part of the form
         * are revised to accommodate location level booking as follows
         */
        describe('VAR-9195 Specification', function() {


            /**
             * 1.     When the user views the page body, the following information is shown:
             *
             * 1.1.   A left margin is shown on the body of the page with the label: Step 1: Type of Care shown
             *
             * 1.2.   To the right of the Step 1 label, instructional text and the type of care field are shown:
             *
             * 1.2.1  Text: The VA Appointment App supports scheduling specific types of care:
             *
             * 1.2.2  Label: Select the type of care you need: (required)
             *
             * 1.2.3  Drop down list box displays all types of care available at the enterprise level (utility
             *        configuration is not applied to filter this list);
             *
             * 1.2.4  Primary care is shown as the first selection in the list, all other list items are shown below
             *        primary care in alphabetical order
             */
            describe('1.0.0 - 1.2.4', function() {
                var view;
                var gas;

                beforeEach(function() {
                    var region = helpers.setupTestingRegion();
                    resourceSpy();
                    fetchSpy(mockData());

                    region.show(new View({
                        model: new Backbone.Model(),
                        googleAnalyticsProvider: {
                            getInstance: function() {
                                gas = jasmine.createSpyObj('gas', ['gas']);
                                return gas;
                            },
                        },
                    }));

                    view = region.currentView;

                });

                afterEach(helpers.setupTestingRegion);

                it('show the correct margin', function() {
                    var legend = view.$('.form-section-heading').text()
                        .trim();
                    var arr = legend.split('\n');
                    var text;

                    arr = _.map(arr, function(item) {
                        return item.trim();
                    });
                    text = arr.join(' ');

                    expect(text).toBe('Step 1: Type of Care');
                });

                it('shows the correct instructions', function() {
                    var instructions = view.$('.step-instructions').text()
                        .trim();
                    expect(instructions).toBe('The VA Appointments app supports scheduling specific types of care. If you do not find the care you need, please contact your facility for assistance.');
                });

                it('creates a dropdown full of type of care items', function() {
                    var options = view.$('#typeOfCare').children();

                    // collection.length + 1 because of the placeholder
                    expect(options.length).toBe(view.collection.length + 1);
                });

                it('always puts primary care the top of the list', function() {
                    var options = view.$('#typeOfCare').children();
                    var first = options[1];
                    var text = first.innerText.trim();

                    expect(text).toBe('Primary Care');
                });

                it('uses alphabetical order after primary care', function() {
                    var options = view.$('#typeOfCare').children();
                    var arr = _.map(options, function(node) {
                        return node.innerText.trim();
                    });

                    var afterPrimary = arr.slice(2);
                    var last = afterPrimary.shift();
                    var isG00d = true;
                    var next;

                    while (afterPrimary.length) {
                        next = afterPrimary.shift();
                        if (next.localeCompare(last) < 1) {
                            isG00d = false;
                            break;
                        }
                        last = next;
                    }

                    expect(isG00d).toBeTruthy();
                });
            });
        });


        describe('Completing Coverage', function() {
            var view;
            var data;
            var gas;

            beforeEach(function() {
                var region = helpers.setupTestingRegion();
                var id;

                resourceSpy();
                data = mockData();
                fetchSpy(data);
                id = data[0].id;

                region.show(new View({
                    model: new Backbone.Model({typeOfCare: id}),
                    googleAnalyticsProvider: {
                        getInstance: function() {
                            gas = jasmine.createSpyObj('gas', ['gas']);
                            return gas;
                        },
                    },
                }));

                view = region.currentView;
            });

            it('gets the correct name for the selected type of care', function() {
                var expected = data[0].name;
                var actually = view.getCareName();

                expect(actually).toBe(expected);
            });

            it('getCareName can handle the case that type is not selected', function() {
                view.model.set('typeOfCare', 'Does Not Exits');

                expect(view.getCareName()).toBe('');
            });

            it('should call Google Analytics when the type of care is changed', function() {
                view.model.set('typeOfCare', 'Audiology');

                expect(gas.gas.calls.first().args).toEqual(['send', 'event', 'veteran-appointment', 'type-of-care-selected', 'Audiology']);
            });
        });
    });
});
T